<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistMainCategoriesExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistMainCategories\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistMainCategoriesImport;
use App\Models\TherapistMainCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistMainCategoriesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist_main_category.index');
        $this->middleware('permission:therapist_main_category.add', ['only' => ['store']]);
        $this->middleware('permission:therapist_main_category.edit', ['only' => ['update']]);
        $this->middleware('permission:therapist_main_category.delete', ['only' => ['destroy']]);
        $this->middleware('permission:therapist_main_category.export', ['only' => ['export']]);
        $this->middleware('permission:therapist_main_category.import', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null)
    {
        if ($req != null) {
            $therapist_main_categories =  TherapistMainCategory::withAll();
            if ($req->trash && $req->trash == 'with') {
                $therapist_main_categories =  $therapist_main_categories->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_main_categories =  $therapist_main_categories->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_main_categories = $therapist_main_categories->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_main_categories = $therapist_main_categories->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_main_categories = $therapist_main_categories->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_main_categories = $therapist_main_categories->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_main_categories = $therapist_main_categories->get();
                return $therapist_main_categories;
            }
            $therapist_main_categories = $therapist_main_categories->get();
            return $therapist_main_categories;
        }
        $therapist_main_categories = TherapistMainCategory::withAll()->orderBy('id', 'desc')->get();
        return $therapist_main_categories;
    }


    /*********View All TherapistMainCategories  ***********/
    public function index(Request $request)
    {
        $therapist_main_categories = $this->getter($request);
        return view('super_admins.therapist_main_categories.index')->with('therapist_main_categories', $therapist_main_categories);
    }

    /*********View Create Form of TherapistMainCategory  ***********/
    public function create()
    {

        return view('super_admins.therapist_main_categories.create');
    }

    /*********Store TherapistMainCategory  ***********/
    public function store(CreateRequest $request)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if (!$request->is_featured) {
                $data['is_featured'] = 0;
            }
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_main_categories');
            $data['icon'] = uploadCroppedFile($request, 'icon', 'therapist_main_categories');

            $therapist_main_category = TherapistMainCategory::create($data);
            $therapist_main_category->slug = Str::slug($therapist_main_category->name . ' ' . $therapist_main_category->id, '-');
            $therapist_main_category->save();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_main_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_main_categories.index')->with('message', 'TherapistMainCategory Created Successfully')->with('message_type', 'success');
    }

    /*********View TherapistMainCategory  ***********/
    public function show(TherapistMainCategory $therapist_main_category)
    {
        return view('super_admins.therapist_main_categories.show', compact('therapist_main_category'));
    }

    /*********View Edit Form of TherapistMainCategory  ***********/
    public function edit(TherapistMainCategory $therapist_main_category)
    {
        return view('super_admins.therapist_main_categories.edit', compact('therapist_main_category'));
    }

    /*********Update TherapistMainCategory  ***********/
    public function update(CreateRequest $request, TherapistMainCategory $therapist_main_category)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if (!$request->is_featured) {
                $data['is_featured'] = 0;
            }
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_main_categories', $therapist_main_category->image);
            } else {
                $data['image'] = $therapist_main_category->image;
            }
            if ($request->icon) {
                $data['icon'] = uploadCroppedFile($request, 'icon', 'therapist_main_categories', $therapist_main_category->icon);
            } else {
                $data['icon'] = $therapist_main_category->icon;
            }
            $therapist_main_category->update($data);
            $therapist_main_category = TherapistMainCategory::find($therapist_main_category->id);
            $slug = Str::slug($therapist_main_category->name . ' ' . $therapist_main_category->id, '-');
            $therapist_main_category->update([
                'slug' => $slug
            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_main_categories.index')->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_main_categories.index')->with('message', 'TherapistMainCategory Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_main_categories = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_main_categories." . $extension;
        return Excel::download(new TherapistMainCategoriesExport($therapist_main_categories), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistMainCategoriesImport, $file);
        return redirect()->back()->with('message', 'Blog Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE TherapistMainCategory ***********/
    public function destroy(TherapistMainCategory $therapist_main_category)
    {
        $therapist_main_category->delete();
        return redirect()->back()->with('message', 'TherapistMainCategory Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE TherapistMainCategory ***********/
    public function destroyPermanently(Request $request, $therapist_main_category)
    {
        $therapist_main_category = TherapistMainCategory::withTrashed()->find($therapist_main_category);
        if ($therapist_main_category) {
            if ($therapist_main_category->trashed()) {
                if ($therapist_main_category->image && file_exists(public_path($therapist_main_category->image))) {
                    unlink(public_path($therapist_main_category->image));
                }
                $therapist_main_category->forceDelete();
                return redirect()->back()->with('message', 'Blog Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Blog Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore TherapistMainCategory***********/
    public function restore(Request $request, $therapist_main_category)
    {
        $therapist_main_category = TherapistMainCategory::withTrashed()->find($therapist_main_category);
        if ($therapist_main_category->trashed()) {
            $therapist_main_category->restore();
            return redirect()->back()->with('message', 'Blog Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Blog Category Not Found')->with('message_type', 'error');
        }
    }
}
